#include "drv_utils.h"


#include <unistd.h>

#include <stdio.h>
#include <string.h>
#include <rtl_printf.h>

//local prototypes
unsigned short scale_dac (double voltage);
static int check_valid_address (int address);

//include one of these defines for debugging output
#define DRV_RTLINUX
//#define DRV_QNX

//defines
#define ADC_POLL_LIMIT 20  //limit depends on target speed

///////////////////////////
//open the board 
//  returns the handle (or zero if failure)
//
int open_board (void)
{
	int fd;

	fd = open("/dev/servotogo",1);

	rtl_printf("servotogo opened  with handle %d\n", fd);

	return fd;
}

//////////////////////////
//close board
//
//
void close_board (int fd)
{
	close(fd);
}

////////////////////
//initialize the board
//	this is currently done from here to support customization
//	on a per app basis, but could be built into the driver
//
int init_board (int fd)
{
	int i;

	//initialize the encoders
	// code borrowed from robotstg.c (original version in comments)
	for (i=0; i<6; i+=2)
	{
		//outport(pumaVarsG.base + ENC_CTRL_BASE + 2*i, ENC_MCR_INIT);
		drv_write_word (fd, ENC_MCR_INIT, ENC_CTRL_BASE + 2*i);

		//outport(pumaVarsG.base + ENC_CTRL_BASE + 2*i, ENC_ICR_INIT);
		drv_write_word (fd, ENC_ICR_INIT, ENC_CTRL_BASE + 2*i);

		//outport(pumaVarsG.base + ENC_CTRL_BASE + 2*i, ENC_OCR_INIT);
		drv_write_word (fd, ENC_OCR_INIT, ENC_CTRL_BASE + 2*i);

		//outport(pumaVarsG.base + ENC_CTRL_BASE + 2*i, ENC_QCR_INIT);
		drv_write_word (fd, ENC_QCR_INIT, ENC_CTRL_BASE + 2*i);
	}

	return 0;
}

/////////////////////////////////////
//write a byte to an address
//	this stuffs data into buffer and sends it to driver
//
int drv_write_byte (int fd, unsigned char data, short address)
{
	static char buffer1[10];
	static char buffer2[5];
	int status;

	sprintf(buffer1, "%d", address);

	sprintf(buffer2, "%d", data);
	strcat(buffer1,",");
	strcat(buffer1,buffer2);
	strcat(buffer1,":0");

	//ready to write
	status = write(fd, buffer1, sizeof(buffer1));

	return status;
}

/////////////////////////////////////
//write a word to an address
//	this stuffs data into buffer and sends it to driver
//
int drv_write_word (int fd, short data, short address)
{
	static char buffer1[10];
	static char buffer2[5];
	int status;

	sprintf(buffer1, "%d", address);

	sprintf(buffer2, "%d", data);
	strcat(buffer1,",");
	strcat(buffer1,buffer2);
	strcat(buffer1,":1");

	//ready to write
	status = write(fd, buffer1, sizeof(buffer1));

	return status;
}

/////////////////////////////////////
//read a byte from an address
//	this stuffs a read address into buffer and sends it to driver
//
unsigned char drv_read_byte (int fd, short address)
{
	static char buffer1[10];
	short result;

	sprintf(buffer1, "%d", address);
	strcat(buffer1,":0");

	//ready to write
	result = read(fd, buffer1, sizeof(buffer1));

	return result;
}

/////////////////////////////////////
//read a word from an address
//	this stuffs a read address into buffer and sends it to driver
//
unsigned short drv_read_word (int fd, short address)
{
	static char buffer1[10];
	short result;

	sprintf(buffer1, "%d", address);
	strcat(buffer1,":1");

	//ready to write
	result = read(fd, buffer1, sizeof(buffer1));

	return result;
}

/////////////////////////////////////
//configures a DIO channel
//  direction parameters (dirX) are for channels A, B, C
//  dirC0 is for the lower 4 bits of C, dirC1 is for the upper four bits
//  dirC0 is for the lower 4 bits of C, dirC1 is for the upper four bits
//  in all cases 1=input, 0=output
void config_dio (int fd, int dirA, int dirB, int dirC0, int dirC1, 
                 int dirD0, int dirD1)
{
  unsigned char write_val_ABC;
  unsigned char write_val_D;

  //init bytes
  //ports ABC and D are configured separately, so we have separate bytes
  write_val_ABC = 0x80;
  write_val_D = 0x82;

  //configure ABC
  if (dirA)
    write_val_ABC |= 0x10;
  if (dirB)
    write_val_ABC |= 0x02;
  if (dirC0)
    write_val_ABC |= 0x01;
  if (dirC1)
    write_val_ABC |= 0x08;
  
  //configure D
  if (dirD0)
    write_val_D |= 0x01;
  if (dirD1)
    write_val_D |= 0x08;

  //write the configuration bytes
  drv_write_byte (fd, write_val_ABC, ABC_DIR);
  drv_write_byte (fd, write_val_D, D_DIR);
}

/////////////////////////////////////
//writes a byte to a DIO channel
//  channel is 0=A, 1=B, 2=C low nibble
//     3=C high nibble, 4=D low, 5=D high
void write_dio (int fd, int channel, unsigned char value)
{
  int offset;

  switch (channel)
  {
    case 0:  
      offset = PORT_A;
      break;
    case 1:  
      offset = PORT_B;
      break;
    case 2:  
      offset = PORT_C;
      break;
    case 3:  
      offset = PORT_C;
      break;
    case 4:  
      offset = PORT_D;
      break;
    case 5:  
      offset = PORT_D;
      break;
    default:  
      offset = 0;
      break;
  }

    if (offset == 0)
      //didn't find an offset.  fail and return
      return;

    drv_write_byte (fd, value, offset);

#ifdef DRV_RTLINUX			
			rtl_printf("wrote dio %d: %d \n", channel, value);
#endif
#ifdef DRV_QNX
#endif

}

/////////////////////////////////////
//reads a byte to a DIO channel
//  channel is 0=A, 1=B, 2=C low nibble
//     3=C high nibble, 4=D low, 5=D high
unsigned char read_dio (int fd, int channel)
{
  int offset;
  unsigned char value;

  switch (channel)
  {
    case 0:  
      offset = PORT_A;
      break;
    case 1:  
      offset = PORT_B;
      break;
    case 2:  
      offset = PORT_C;
      break;
    case 3:  
      offset = PORT_C;
      break;
    case 4:  
      offset = PORT_D;
      break;
    case 5:  
      offset = PORT_D;
      break;
    default:  
      offset = 0;
      break;
  }

    if (offset == 0)
      //didn't find an offset.  fail and return
      return 0 ;

    value = drv_read_byte (fd, offset);

#ifdef DRV_RTLINUX			
    rtl_printf("read dio %d: %d \n", channel, value);
#endif
#ifdef DRV_QNX
#endif

    return value;
}

/////////////////////////////////////
//outputs a voltage on the specified DAC channel
//  figures out the address offset to write to from the channel
//
void output_dac (int fd, int channel, double voltage)
{
	unsigned short dac_counts = 0;
	int offset = 0;

	switch (channel)
	{
		case 0:
			offset = DAC_0;
			break;
		case 1:
			offset = DAC_1;
			break;
		case 2:
			offset = DAC_2;
			break;
		case 3:
			offset = DAC_3;
			break;
		case 4:
			offset = DAC_4;
			break;
		case 5:
			offset = DAC_5;
			break;
		case 6:
			offset = DAC_6;
			break;
		case 7:
			offset = DAC_7;
			break;
		default:
#ifdef DRV_RTLINUX			
			rtl_printf("bad DAC channel: %d \n", channel);
#endif
#ifdef DRV_QNX

#endif
			break;
			return;
	}
	
	dac_counts = scale_dac (voltage);

	drv_write_word (fd, dac_counts, offset);
}

/////////////////////////////////////
//inputs a voltage from a specified ADC channel
//  figures the address from the channel and the board's 
//returns the voltage as a double
//
double input_adc (int fd, int channel)
{
	unsigned char config = 0;
	unsigned short data_read = 0;
	int end_conversion = 1;

  int count;
	float tempf;
	unsigned short temps;
	int tempi;

	//in all cases, CNTRL0 register bits match the channel #
	//bits start at bit 5 of the register
	config = channel;
	config = config << 4;

	//disable cal
	config |= CNTRL0_CAL;

	//disable autozero for faster conversions
	config |= CNTRL0_AZ;

	//select the channel
	drv_write_byte (fd, config, CNTRL0);

	//tbd wait 4 usec for settling time
	//this may or may not be necessary, probably depending on the target
	for (tempi = 0; tempi <1000; tempi++);

	//start conversion - any 16 bit data write will work
	drv_write_word (fd, 0x0000, ADC);

	//wait for conversion to finish
  count = 0;
	while((0 != end_conversion) && (count < ADC_POLL_LIMIT))
  {
		data_read = drv_read_byte (fd, BRDTST);
		end_conversion = data_read & EOC;
    count++;
  }

  //check for failure
  if (count >= ADC_POLL_LIMIT)
  {
#ifdef DRV_RTLINUX			
	  rtl_printf("ADC read failed.");
	  return -100.0f;
#endif
  }

	//conversion finished.  get the data
	data_read = drv_read_word (fd, ADC);

	//convert from 2's complement and scale counts to voltage
	temps = data_read & 0x1FFF;
	if (temps & 0x1000)
	{
		//negative
		temps = temps & 0x0FFF;
		temps = 0xFFF - temps + 1;
		
		//assume 10 volt range.  1 count = 0.0024 volts	
		tempf = temps * -0.0024f;
	}
	else
	{
		//positive
		tempf = temps * 0.0024f;
	}


#ifdef DRV_RTLINUX			
	tempi = tempf * 1000;
	rtl_printf("ADC read.  channel %d raw = 0x%x  mV = %d\n", channel, data_read, tempi);
#endif

#ifdef DRV_QNX

#endif

	return (double)tempf;

}

/////////////////////////////////////
//inputs a count from a specified encoder channel
//  figures the address from the channel and the board's base
//returns the count as a double or -1 for a bad channel
//
double input_encoder (int fd, int chan)
{
	double output_count;
	int i;

	unsigned char  tempEnc[2];
    long  tempLong[2];

	//check channel
	if ((chan < 0) || (chan > 7))
		return -1;

	//read the encoder
	//  code borrowed from robotstg.c

	//encoder values are arranged in pairs to facilitate word reads
	//first, figure offset i - for odd channels, determine offset based
	// on next lowest even channel (1 bit shift throws away LSB)

	i = chan >> 1;
	i = i << 1;

	/* Reset the address pointer and transfer to output latch */
	//outport(pumaVarsG.base + ENC_CTRL_BASE + 2*i, ENC_MCR_READ);
	drv_write_word (fd, ENC_MCR_READ, ENC_CTRL_BASE + 2*i);

	/* Read the first byte of a pair of channels */
	//*((short *)tempEnc) = inport(pumaVarsG.base + ENC_COUNT_BASE + 2*i);
	*((short *)tempEnc) = drv_read_word (fd, ENC_COUNT_BASE + 2*i);

	/* Shift everything up 8 bits for sign extension */
	tempLong[0] = (long)tempEnc[0] << 8;
	tempLong[1] = (long)tempEnc[1] << 8;

	/* Read the middle byte of 2 adjacent channels */
	//*((short *)tempEnc) = inport(pumaVarsG.base + ENC_COUNT_BASE + 2*i);
	*((short *)tempEnc) = drv_read_word (fd, ENC_COUNT_BASE + 2*i);

	tempLong[0] |= (long)tempEnc[0] << 16;
	tempLong[1] |= (long)tempEnc[1] << 16;

	/* Read the high byte of two adjacent channels */
	//*((short *)tempEnc) = inport(pumaVarsG.base + ENC_COUNT_BASE + 2*i);
	*((short *)tempEnc) = drv_read_word (fd, ENC_COUNT_BASE + 2*i);

	tempLong[0] |= (long)tempEnc[0] << 24;
	tempLong[1] |= (long)tempEnc[1] << 24;
	tempLong[0] /= 256;
	tempLong[1] /= 256;

	//determine which of the pair of counts that was read was the one
	//  we wanted - even or odd
	if (i == chan)
// tbd skip scaling for now	
//		val = encoderScaleG[chan] * (tempLong[0] - encoderOffsetG[chan]) +
//		jointOffsetG[chan];
    output_count = (double)tempLong[0];
	else
//tbd skip scaling for now
//		val = encoderScaleG[chan] * (tempLong[1] - encoderOffsetG[chan]) +
//		jointOffsetG[chan];
	output_count = (double)tempLong[1];

#ifdef DRV_RTLINUX			
	rtl_printf("Encoder channel %d count value: %x \n", chan, (int)output_count);
#endif

#ifdef DRV_QNX

#endif

	return output_count;

}


/////////////////////////////////////
//presets a count to a specified encoder channel 
//  figures the address from the channel and the board's base
//returns 0 for success and -1 for a bad channel
//
double output_encoder (int fd, int chan, short val)
{
	short i,j,tempEnc;

	//check channel
	if ((chan < 0) || (chan > 7))
		return -1;

	//write the encoder
	//  code borrowed from robotstg.c

	//encoder values are arranged in pairs to facilitate word reads
	//first, figure offset i - for odd channels, determine offset based
	// on next lowest even channel (1 bit shift throws away LSB)

	i = chan & 0x7FFE;

	j = chan & 0x0001;

	/* Reset the address pointer and transfer to output latch */
	//outport(pumaVarsG.base + ENC_CTRL_BASE + 2*i, ENC_MCR_READ);
	drv_write_word (fd, ENC_MCR_READ, ENC_CTRL_BASE + 2*i);

	/* Ready the low byte of one channel */
//tbd should this use encoderOffsetG ?
//	tempEnc = (val + encoderOffsetG[chan]) & 0x00FF;
	tempEnc = val & 0x00FF;

	//outportb(pumaVarsG.base + ENC_LOAD_BASE + 2*i+j, (char)tempEnc);
	drv_write_byte (fd, (unsigned char)tempEnc, ENC_LOAD_BASE + 2*i+j);

	/* Ready the middle byte of one channel */
//tbd should this use encoderOffsetG?
//	tempEnc = ((val + encoderOffsetG[chan]) >> 8) & 0x00FF;
	tempEnc = (val >> 8) & 0x00FF;

	//outportb(pumaVarsG.base + ENC_LOAD_BASE + 2*i+j, tempEnc);
	drv_write_byte (fd, (unsigned char)tempEnc, ENC_LOAD_BASE + 2*i+j);

	/* Ready the high byte of one channel */
//tbd should this use encoderOffsetG?
	//tempEnc = (short)((((int)val + (int)encoderOffsetG[chan]) >> 16) & 0x00FF);
	tempEnc = (val  >> 16) & 0x00FF;
	
	//outportb(pumaVarsG.base + ENC_LOAD_BASE + 2*i+j, tempEnc);
	drv_write_byte (fd, (unsigned char)tempEnc, ENC_LOAD_BASE + 2*i+j);

	/* Transfer the preload to the counter */
	//outportb(pumaVarsG.base + ENC_CTRL_BASE + 2*i+j, (char)(ENC_MCR_LOAD));
	drv_write_byte (fd, (unsigned char)(ENC_MCR_LOAD), ENC_CTRL_BASE + 2*i+j);

#ifdef DRV_RTLINUX			
	rtl_printf("Encoder channel %d write value: %d \n", chan, val);
#endif

#ifdef DRV_QNX

#endif

	return 0;

}
/////////////////////////////////////
//formats a dac voltage
//  takes a voltage -10 to 10 as a double
//  returns a short int scaled between 0x0000(-10v) and 0x1fff (10v)
unsigned short scale_dac (double voltage)
{
	double tempd;

	tempd = voltage + 10;  //get rid of offset
	tempd = tempd * 409.6f;  //scale to dac counts

	if (tempd > 8191)
		tempd = 8191;

	return (unsigned short)tempd;

}


///////////////////////////////////////
//checks the validity of a servotogo board address
//
static int check_valid_address (int address)
{
	int addresses[16] = {	0x200, 0x220, 0x240, 0x260, 0x280,
					  		0x2A0, 0x2C0, 0x2E0, 0x300, 0x320,
							0x340, 0x360, 0x380, 0x3A0, 0x3C0,
							0x3E0 };
	int i;

	for (i=0; i<15; i++)
	{
		if (addresses[i] == address)
			return 1;
	}

	return 0;

}
