package I2A2::TicketTool;

# Based on the TicketMaster module from:
#    "Writing Apache Modules with Perl and C", Stein & MacEachern

=head1 NAME

I2A2::TicketTool - Routines for dealing with tickets.

=head1 SYNOPSIS

use I2A2::TicketTool;

$ticketTool = I2A2::TicketTool->new($r);

$ticket = $ticketTool->make_ticket($r, $alias, $puid, $realm);

($result, $msg, $ticketInfo) = $ticketTool->verify_ticket($r);

$url = $ticketTool->loginURL;

$cookie = $ticketTool->make_return_address($r);

fetch_secret
invalidate_secret


TicketExpires
TicketSecret
TicketDomain
TicketLoginURL
TicketName

=head1 DESCRIPTION

...

=cut

use strict;
use CGI::Cookie ();
use Digest::MD5 ();
#use LWP::Simple ();
use Apache::Constants qw(:common);
use Apache::File ();
use Apache::URI ();

if ($ENV{MOD_PERL}) {
    my $ServerName = Apache->server->server_hostname;
} else {
    my $ServerName = "TESTING";
}

my %DEFAULTS = (
#   'TicketDatabase' => 'mysql:test_www',
#   'TicketTable'    => 'user_info:user_name:passwd',
    'TicketExpires'  => 30,
    'TicketSecret'   => 'http://$ServerName/secret_key.txt',
    'TicketDomain'   => undef,
    'TicketLoginURL' => undef,
    'TicketName'     => 'I2A2_Authentication',
);

my %CACHE;  # cache objects by their parameters to minimize time-consuming operations

# Set up default parameters by passing in a request object
sub new {
    my($class, $r) = @_;
    my %self = ();
    foreach (keys %DEFAULTS) {
	$self{$_} = $r->dir_config($_) || $DEFAULTS{$_};
    }
    # post-process TicketDomain
    ($self{TicketDomain} = $r->server->server_hostname) =~ s/^[^.]+// 
	unless $self{TicketDomain};

    # try to return from cache
    my $id = join '', sort values %self;
    return $CACHE{$id} if $CACHE{$id};

    # otherwise create new object
    return $CACHE{$id} = bless \%self, $class;
} 

# TicketTool::fetch_secret()
# Call as:
# $ticketTool->fetch_secret();
sub fetch_secret {
    my $self = shift;
    unless ($self->{SECRET_KEY}) {
	if ($self->{TicketSecret} =~ /^http:/) {
	    $self->{SECRET_KEY} = LWP::Simple::get($self->{TicketSecret});
	} else {
	    my $fh = Apache::File->new($self->{TicketSecret}) || return;
	    $self->{SECRET_KEY} = <$fh>;
	    $fh->close();
	}
    }
    $self->{SECRET_KEY};
}

# return the URL for the Ticket Login Screen
sub loginURL {
    my($self) = shift;
    return ($self->{TicketLoginURL}) if ($self->{TicketLoginURL});
    return;	# not specified
}

# invalidate the cached secret
sub invalidate_secret { undef shift->{SECRET_KEY}; }

# TicketTool::make_ticket()
# Call as:
# $cookie = $ticketTool->make_ticket($r, $user, $puid, $realm);
#
sub make_ticket {
    my($self, $r, $user_name, $puid, $realm) = @_;
    my $ip_address = $r->connection->remote_ip;
    my $expires = $self->{TicketExpires};
    my $now = time;
    my $secret = $self->fetch_secret() or return undef;
    my $hash = Digest::MD5::md5_hex($secret .
		    Digest::MD5::md5_hex(join ':', $secret, $ip_address, $now,
			$expires, $user_name, $puid, $realm)
	       );
    return CGI::Cookie->new(-name => $self->{TicketName},
			    -path => '/',
			    -domain => $self->{TicketDomain},
			    -value => {
				'ip'      => $ip_address,
				'time'    => $now,
				'realm'   => $realm,
				'user'    => $user_name,
				'puid'    => $puid,
				'hash'    => $hash,
				'expires' => $expires,
			    });
}


# TicketTool::verify_ticket()
# Call as:
# ($result,$msg) = $ticketTool->verify_ticket($r)
#    returned $result will be OK or AUTH_REQUIRED
sub verify_ticket {
    my($self, $r) = @_;
    my %cookies = CGI::Cookie->parse($r->header_in('Cookie'));
    return (AUTH_REQUIRED, 'user has no cookies', undef) unless %cookies;
    return (AUTH_REQUIRED, 'user has no ticket', undef)
	unless $cookies{$self->{TicketName}};
    my %ticket = $cookies{$self->{TicketName}}->value;
    return (AUTH_REQUIRED, 'malformed ticket', undef)
	unless $ticket{'hash'} && $ticket{'user'} && $ticket{'puid'} &&
	    $ticket{'realm'} && $ticket{'time'} && $ticket{'expires'};
    return (AUTH_REQUIRED, 'IP address mismatch in ticket', undef)
	unless $ticket{'ip'} eq $r->connection->remote_ip;
    return (AUTH_REQUIRED, 'ticket has expired', undef)
	unless (time - $ticket{'time'})/60 < $ticket{'expires'};
    my $secret;
    return (AUTH_REQUIRED, "can't retrieve secret", undef)
	unless $secret = $self->fetch_secret;
    my $newhash = Digest::MD5::md5_hex($secret .
			Digest::MD5::md5_hex(join ':', $secret,
			    @ticket{qw(ip time expires user puid realm)})
		  );
    unless ($newhash eq $ticket{'hash'}) {
	$self->invalidate_secret;  #maybe it's changed?
	return (AUTH_REQUIRED, 'ticket mismatch', undef);
    }
    $r->connection->user($ticket{'user'});
    return (OK, 'ok', {%ticket});		# anon hash reference
}

# Call as:
# $cookie = $ticketTool->make_return_address()
sub make_return_address {
    my($self, $r) = @_;
    my $link = Apache::URI->parse($r)->unparse;		# rebuild the request

    return CGI::Cookie->new(-name => 'request_uri',
			    -value => $link,
			    -domain => $self->{TicketDomain},
			    -path => '/');
}

1;
__END__

=head1 AUTHOR

Jeff W. Stewart, jws@purdue.edu

=head1 SEE ALSO

=cut
