package I2A2::TicketLogin;

# Based heavily on the TicketMaster module from:
#    "Writing Apache Modules with Perl and C", Stein & MacEachern

=head1 NAME

I2A2::TicketLogin - Apache handlers for I2A2 Login.

=head1 SYNOPSIS

 PerlModule I2A2::TicketLogin;

 PerlSetVar  TicketDomain    <domain>
 PerlSetVar  TicketSecret    <secret_text_path>
 PerlSetVar  TicketExpires   <timeout>

 <Location /ticketLogin>
     SetHandler         perl-script
     PerlHandler        I2A2::TicketLogin
     PerlSetVar         PurdueCertURL   <URL for PurdueCertLogin>
 </Location>

 <Location /PurdueCertLogin>
     SSLVerifyClient    require
     SSLVerifyDepth     2
     SSLOptions         +ExportCertData +StdEnvVars
     SSLRequireSSL
     PerlAuthenHandler  Apache::OK
     AuthName           SSL
     AuthType           Basic
     SetHandler         perl-script
     PerlHandler        I2A2::TicketLogin::PurdueCertLogin
 </Location>


=head1 DESCRIPTION

The I2A2::TicketLoging module contains the Apache handlers dealing with
athenticating a user, one via password access, the other via a Purdue
issued X509 certificate.

Both handlers should only be used in the Secure Web Server (SSL).

=over 4

=cut

use strict;
use Apache::Constants qw(:common);
use Apache::Log ();
use I2A2;
use I2A2::Authc;
use I2A2::DB;
use I2A2::TicketTool ();
use CGI '-autoload';

=item I2A2::TicketLogin::handler

The main handler presents a log-in form and processes log-in requests.
The log-in form includes a pull-down menu to specify a realm, text boxes
for alias and password, a log-in button, and a button to supply a Purdue
certificate.

When the log-in button is selected, authentication is attempted with the
supplied information.  If successful, an authentication ticket is created
and stored in a Cookie.  Then, a redirect is issued for the originally
reaquested URL.  Otherwise, the log-in form is re-displayed.

When the "Supply a Purdue Certificate" button is selected, a redirect is
issued to a URL where an SSL Client Certificate will be requested (defined
by the PurdueCertLogin variable).

=cut

# This is the log-in screen that provides authentication cookies.
# There should already be a cookie named "request_uri" that tells
# the login screen where the original request came from.
sub handler {
    my $r = shift;
    my (@realms, $realm_default);
    my($action, $user, $pass, $realm) =
	map { param($_) } qw(action user password realm);
    my $request_uri = param('request_uri') || 
	($r->prev ? $r->prev->uri : cookie('request_uri'));

    unless ($request_uri) {
	no_cookie_error();
	return OK;
    }

    my($err, @ans);
    if ($action) {
	if ($action eq "Log In" and $user and $pass and $realm) {
	    ($err, @ans) = Authenticate($user, $pass, $realm);
	    if ($err == 0) {
		my ($puid, $alias, $name) = @ans;
		my $ticketTool = I2A2::TicketTool->new($r);
		my $ticket = $ticketTool->make_ticket($r, $alias, $puid, $realm);
		unless ($ticket) {
		    $r->log_error("Couldn't make ticket -- missing secret?");
		    return SERVER_ERROR;
		}
		go_to_uri($r, $request_uri, $ticket, $puid, $alias, $name, $realm);
		return OK;
	    }
	    $ans[0] = "Authentication failed.";
	} elsif ($action eq "Supply Purdue Certificate") {
	    my $cert_url = $r->dir_config('PurdueCertURL');
	    unless ($cert_url) {
		$r->log_error("Couldn't get URL from config (PurdueCertURL)");
		return SERVER_ERROR;
	    }
	    print header(-refresh => "1; URL=$cert_url"),
	    start_html(-title => 'Purdue Certificate Login', -bgcolor => 'gold'),
	    h1('Purdue University Identification'),
	    h2('Please supply a Purdue Identification Certificate'),
	    end_html();
	    return OK;
	} else {
	    $ans[0] = "Unknown action ($action).";
	}
    }

    my $realmlist = param('realmlist');
    if (defined $realmlist) {
	@realms = split /,/, $realmlist;
    } else {
	@realms = ('Purdue');			# Default realm list
    }
    $realm_default = shift @realms;

    make_login_screen($ans[0], $request_uri, $realm_default, \@realms);

    return OK;
}

=item I2A2::TicketLogin::PurdueCertLogin

This handler deals with Purdue issued certificates.  The user's PUID
is extracted from the certificate and verified with the Authentication
DBM.

If successful, an authentication ticket is created and stored in a Cookie.
Then, a redirect is issued for the originally reaquested URL.  Otherwise,
an "Authentication Failed" page is displayed.

=cut

#
# handler for the PurdueCertLogin page.
#

sub PurdueCertLogin {
    my $r = shift;
    my ($err, $msg);
    my ($pem, $x509, $puid);
    my (%ans);

    return OK unless $r->is_main;	# nothing needed for subrequests

    my $request_uri = param('request_uri') ||
	($r->prev ? $r->prev->uri : cookie('request_uri'));

    # get the client certificate by invoking a subrequest
    my $subr = $r->lookup_uri($r->uri);
    $pem = $subr->subprocess_env('SSL_CLIENT_CERT');
    unless ($pem) {
	$msg = "Couldn't get PEM certificate";
	$err = SERVER_ERROR;
    }
    if (!$err && I2A2::libpuidX::X509fromPEM($pem, $x509)) {
	$msg = "Couldn't convert PEM to X509";
	$err = SERVER_ERROR;
    }
    if (!$err && I2A2::libpuidX::puid_puissuer($x509)) {
	$msg = "Certificate not issued by proper authority";	
	$err = FORBIDDEN;
    }
    if (!$err && I2A2::libpuidX::puid_getcertPUID($x509, $puid)) {
	$msg = "Couldn't extract PUID from Certificate";
	$err = FORBIDDEN;
    }

    if (!$err) {				# no errors so far?
	my $ans = RealmLookup($puid, "purdue");
	%ans = I2A2::DB::DBparse($ans);

	if ($ans{REPLY} ne &I2A2::PUIDNETD_REPL_ACK) {
	    $msg = "Realm lookup failed\n";
	    foreach (@{$ans{&I2A2::PUIDNETD_DATA_MSG}}) {
		$msg .= "<br>$_\n";
	    }
	    $err = FORBIDDEN;
	} elsif (!defined $ans{&I2A2::PUIDNETD_DATA_AUTHC_REC}) {
	    $msg = "Realm lookup failed";
	    $err = FORBIDDEN;
	} else {
	    my $ra = $ans{&I2A2::PUIDNETD_DATA_AUTHC_REC};
	    my %recfields = I2A2::DB::DBparseFields($ra->[0]);
	    if (!defined $recfields{&I2A2::PUIDNETD_DATA_AUTHC_CERT}) {
		$msg = "Purdue certificate not registered";
		$err = FORBIDDEN;
	    } else {
		$pem =~ m/--\n([^-]+)\n--/;		# extract base64 cert
		my ($cert) = $1;
		$cert =~ s/\n//g;
		if ($cert ne $recfields{&I2A2::PUIDNETD_DATA_AUTHC_CERT}) {
		    $msg = "Certificate could not be verified";
		    $err = FORBIDDEN;
		}
	    }
	}
    }
    if (!$err) {				# no errors so far?
	my $puid = $ans{&I2A2::PUIDNETD_DATA_PUID};
	my $alias = $ans{&I2A2::PUIDNETD_DATA_AKA};
	my $name = $ans{&I2A2::PUIDNETD_DATA_CNM};
	my $ticketTool = I2A2::TicketTool->new($r);
	my $ticket = $ticketTool->make_ticket($r, $alias, $puid, "Purdue");
	unless ($ticket) {
	    $r->log_error("Couldn't make ticket -- missing secret?");
	    return SERVER_ERROR;
	}
	go_to_uri($r, $request_uri, $ticket, $puid, $alias, $name, "Purdue");
	return OK;
    }
    my $explanation = <<END;
<TITLE>Authentication Failed</Title>
<H1>You are not authorized to access this page</H1>
<H3>Authentication Failed.</H3>
END
    $explanation .= $msg;
    $r->custom_response($err, $explanation);
    $r->log_reason($msg, $r->filename);
    return $err;
}

sub go_to_uri {
    my($r, $requested_uri, $ticket, $puid, $alias, $name, $realm) = @_;
    print header(-refresh => "1; URL=$requested_uri", -cookie => $ticket),
    start_html(-title => 'Successfully Authenticated', -bgcolor => 'white'),
    h1('Congratulations'),
    h2('You have successfully authenticated'),
    h4("Realm: $realm"),
    h4("Alias: $alias"),
    h4("PUID:  " . I2A2::Format_PUID($puid)),
    h4("Name:  $name"),
    h3("Please stand by..."),
    end_html();
}

sub make_login_screen {
    my($msg, $request_uri, $realm_default, $ra_realms) = @_;
    print header(),
    start_html(-title => 'Log In', -bgcolor => 'white'),
    h1('Please Log In');
    print  h2(font({color => 'red'}, "Error: $msg")) if $msg;
    print start_form(-action => script_name()),
    table(
	  Tr(td(['Realm', popup_menu('realm', $ra_realms, $realm_default)])),
	  Tr(td(['Login', textfield(-name => 'user')])),
	  Tr(td(['Password', password_field(-name => 'password')]))
	  ),
	      hidden(-name => 'request_uri', -value => $request_uri),
	      hidden(-name => 'realmlist', -value => join(",", $realm_default, @$ra_realms)),
	      submit(-name => 'action', -value => 'Log In'),
	      submit(-name => 'action', -value => 'Supply Purdue Certificate'),
	      p(),
	      end_form(),
	      em('Note: '),
	      "You must set your browser to accept cookies in order for login to succeed.",
	      "You will be asked to log in again after some period of time has elapsed.";
}

# called when the user tries to log in without a cookie
sub no_cookie_error {
    print header(),
    start_html(-title => 'Unable to Log In', -bgcolor => 'white'),
    h1('Unable to Log In'),
    "This site uses cookies for its own security.  Your browser must be capable ", 
    "of processing cookies ", em('and'), " cookies must be activated. ",
    "Please set your browser to accept cookies, then press the ",
    strong('reload'), " button.", hr();
}

1;
__END__

=back

=head1 AUTHOR

Jeff W. Stewart, jws@purdue.edu

=head1 SEE ALSO

=cut
