package I2A2::Ident;

=head1 NAME

I2A2::Ident - I2A2 Identification routines.

=head1 SYNOPSIS

use I2A2::Ident;

($err, @array) = Identify('jws');

=head1 DESCRIPTION

The I2A2::Ident module includes procedures which access the I2A2 Reflector
DBM.

=over 4

=cut

require   Exporter;
@ISA    = qw(Exporter);
@EXPORT = qw(Identify);
@EXPORT_OK = qw();

use strict;

use I2A2;
use I2A2::libpuidX;
use I2A2::DB;

=item ($err, @array) = Identify('jws');

Identify() looks up the given id (PUID, alias, or name) in the reflector
DBM.

If the lookup succeeds, $err will be zero (0) and @array will contain the 
PUID, alias, and name of the user.

On failure, $err will be non-zero and @array will contain strings explaining
the failure.

Example:

    use I2A2;
    use I2A2::Ident;
    ...

    ($err, @array) = Identify("usera");
    if ($err) {
	foreach (@array) {
	    print STDERR "$_\n";
	}
	...	# return error or die, etc.
    }
    ($puid, $alias, $name) = @array;
    ...

=cut

sub Identify {
    my $id = shift;
    my ($err, $line, $req, @ans, %fields);
    my ($puid);


    undef @ans;
    $err = -1;
    $req = &I2A2::PUIDNETD_CMD_LOOKUP;
    if ($id =~ /^\d[-\d]+$/) {		# all digits and "-" (PUID)
	$id =~ s/-//g;			# remove "-" characters
	$req .= &I2A2::PUIDNETD_DATA_PUID . $id . &I2A2::PUIDNETD_MSGTERM;
	if (I2A2::luhnck($id)) {
	    $err =-3;
	    $ans[0] = "Invalid PUID.";
	    goto EXIT;
	}
    } elsif ($id =~ /[\s*]/) {		# contains a space (Name)
	$req .= &I2A2::PUIDNETD_DATA_CNM . $id . &I2A2::PUIDNETD_MSGTERM;
    } else {				# (alias)
	$req .= &I2A2::PUIDNETD_DATA_AKA . $id . &I2A2::PUIDNETD_MSGTERM;
    }

    # This works the hard way by opening a connection for every lookup.
    # Open a connection to the server.

    my $Refl = I2A2::DB->connect(&I2A2::PUIDNETD_HOST_REFL,
		    &I2A2::PUIDNETD_SVC_SSL_REFL, &I2A2::PUIDNETD_REFL_PUID,
		    &I2A2::PUIDNETD_REFL_CSER);
    ### Check for error - JWS

    # Issue the request

    $line = $Refl->request($req);

    %fields = I2A2::DB::DBparse($line); 

    if ($fields{REPLY} eq &I2A2::PUIDNETD_REPL_ACK) {
	$err = 0;
	$ans[0] = $fields{&I2A2::PUIDNETD_DATA_PUID}+0;	# remove leading zeros
	$ans[1] = $fields{&I2A2::PUIDNETD_DATA_AKA};
	$ans[2] = $fields{&I2A2::PUIDNETD_DATA_CNM};
    } elsif ($fields{REPLY} eq &I2A2::PUIDNETD_REPL_NAK) {
	$err = $fields{&I2A2::PUIDNETD_DATA_ERRC};
	@ans = @{$fields{&I2A2::PUIDNETD_DATA_MSG}};
    } else {
	$err = -2;
	$ans[0] = "unexpected answer '$line'.";
    }

    # Close the connection

    $Refl->close();
    undef $Refl;

    EXIT:
    if (wantarray) {
	return ($err, @ans);
    } else {
	return $err;
    }
}

1;
__END__

=back

=head1 AUTHOR

Jeff W. Stewart, jws@purdue.edu

=head1 SEE ALSO

=cut
