package I2A2;

=head1 NAME

I2A2 - Makes fundamental I2A2 procedures and constants available.

=head1 SYNOPSIS

use I2A2;

$string = Format_PUID($puid);

$bool = luhnck($puid);

$puid = luhngen($int);

=head1 DESCRIPTION

The I2A2 module provides access to the constants defined in F<puidnetd.ph>.
These constants are defined within the I2A2 namespace
(e.g. &I2A2::PUIDNETD_DATA_AKA or &I2A2::PUIDNETD_HOST_AUTHC).

It also provides some common procedures:

=over 4

=cut

require   Exporter;
@ISA    = qw(Exporter);
@EXPORT = qw(Format_PUID luhnck luhngen);
@EXPORT_OK = qw();

use strict;

require q(I2A2/puidnetd.ph);

$I2A2::VERSION='0.83';

=pod

=item C<$str = Format_PUID($puid)>

Format_PUID() formats a PUID as a string in the standard PUID format
(00000-00000) and returns the resulting string.

=cut

sub Format_PUID {
	my ($puid) = @_;
	my $MAX_PUID_DIGITS = 10;
	my $s;

	$puid =~ s/-//g;
	$s = sprintf("%0*d", $MAX_PUID_DIGITS, $puid);
	return(substr($s, 0, 5) . "-" . substr($s, 5, 5));
}

=pod

=item C<$bool = luhnck($puid)>

Luhnck() tests the validity of a PUID check digit.  It returns B<TRUE> when
the check digit is invalid and B<FALSE> otherwise.

Example:

    if (luhnck($puid)) {
	print "Invalid PUID: $puid\n";
    }

=cut

sub luhnck {
    my $num = shift;
    my ($len, $i, $ch, $ich, $sum);

    $len = length($num);
    if ($len < 1) {
	return 1;	# Error
    }
    for ($i = 1, $sum = 0; $i <= $len; $i++) {
	$ch = int(substr($num, $len - $i, 1));
	$ich = int($i * $ch);
	while ($ich > 0) {
	    $sum += $ich % 10;
	    $ich = int($ich / 10);
	}
    }
    return (($sum % 10) ? 1 : 0);
}

=pod

=item C<$puid = lunhgen($int)>

Luhngen() generates a valid check digit for the value passed in,
appends it to that value, and returns the result.

=cut

sub luhngen {
    $_ = shift;
    my ($i, $len, $num, $sign, $sum);

    if (/^([+-])(\d+)$/) { $sign = $1; $num = $2; }
    elsif (/^(\d+)$/)    { $sign = ""; $num = $1; }
    else {
	return;
    }
    $len = length($num);
    if ($len < 1) {
	return;
    }
    for ($i = 1, $sum = 0; $i <= $len; $i++) {
	my $ch = int(substr($num, $len - $i, 1));
	my $ich = int(($i + 1) * $ch);
	while ($ich > 0) {
	    $sum += $ich % 10;
	    $ich = int($ich / 10);
	}
    }
    my $ckd = (10 - ($sum % 10)) % 10;
    return "$sign$num$ckd";
}

1;
__END__

=back

=head1 AUTHOR

Jeff W. Stewart, jws@purdue.edu

=head1 SEE ALSO

=cut
