package I2A2::DB;

=head1 NAME

I2A2::DB - I2A2 DBM Access Routines

=head1 SYNOPSIS

use I2A2::DB;

$I2A2::DB::CAdir = "dir-path-of-CA-certs";
$I2A2::DB::CApath = "path-of-CA-certificate";

$conn = I2A2::DB->connect($host, $service, $serverPUID, $serverCSER,
	$publicCert, $privateKey);

$errs = $conn->log_errors($doit, $prefix);

$reply = $conn->request($dbm_command);

$conn->close();

%fields = I2A2::DBparse($reply);

%fields = I2A2::DBparseFields($sub_record);

=head1 DESCRIPTION

...

=cut

require   Exporter;
@ISA    = qw(Exporter);
@EXPORT = qw(DBparse DBparseFields);
@EXPORT_OK = qw();

use vars qw($VERSION);
$VERSION = '1.30';

use strict;
no strict 'refs';

use Net::SSLeay qw(print_errs);
use I2A2;
use I2A2::libpuidX;
use Socket;

Net::SSLeay::load_error_strings();
Net::SSLeay::SSLeay_add_ssl_algorithms();
Net::SSLeay::randomize('/etc/passwd');

$Net::SSLeay::trace = 0;

# SSL CA directory and path
$I2A2::DB::CAdir = &I2A2::PUIDNETD_PUB_CERTS if !defined($I2A2::DB::CAdir);
$I2A2::DB::CApath = '' if !defined($I2A2::DB::CApath);

# setup defaults for service ports
%I2A2::DB::default = (
	&I2A2::PUIDNETD_SVC_SSL_AUTHC() => &I2A2::PUIDNETD_PORT_SSL_AUTHC(),
 	&I2A2::PUIDNETD_SVC_SSL_AUTHZ() => &I2A2::PUIDNETD_PORT_SSL_AUTHZ(),
	&I2A2::PUIDNETD_SVC_SSL_REFL() => &I2A2::PUIDNETD_PORT_SSL_REFL(),
	&I2A2::PUIDNETD_SVC_AUTHC() => &I2A2::PUIDNETD_PORT_AUTHC(),
	&I2A2::PUIDNETD_SVC_AUTHZ() => &I2A2::PUIDNETD_PORT_AUTHZ(),
	&I2A2::PUIDNETD_SVC_REFL() => &I2A2::PUIDNETD_PORT_REFL(),
    );


sub connect {
    my ($class, $serverHost, $serverSVC, $serverPUID, $serverCSER, $ClPub, $ClPriv) = @_;
    my %self = ();
    my ($dest_ip, $dest_port, $dest_serv_params);
    my ($req, $resp, $errs);

    $self{CONNECTED} = 0;

    # Open a connection to the server.

    $dest_ip = gethostbyname($serverHost);
    $dest_port = getservbyname($serverSVC, "tcp");
    unless ($dest_port > 0) {
	if (defined($I2A2::DB::default{$serverSVC})) {
            $dest_port = $I2A2::DB::default{$serverSVC};
	    $self{WARN} .= "Lookup sevice ($serverSVC) failed: using port $dest_port\n";
	} else {
	    $self{ERROR} .= "Lookup sevice ($serverSVC) failed\n";
	    return bless \%self, $class;		# ERROR return
	}
    }
    $dest_serv_params = sockaddr_in($dest_port, $dest_ip);

    $self{socket} = 'S' . $serverSVC;

    if ( ! socket($self{socket}, &AF_INET, &SOCK_STREAM, 0)){
	$self{ERROR} .= "socket() failed\n";
	return bless \%self, $class			# ERROR return
    }
    if ( ! connect($self{socket}, $dest_serv_params)) {
	$self{ERROR} .= "connect() failed: $!\n";
	I2A2::DB::close(\%self);
	return bless \%self, $class;			# ERROR return
    }
    select ($self{socket}); $| = 1; select (STDOUT);	# Eliminate STDIO buffering

    # The network connection is now open, build an SSL context

    $self{ctx} = Net::SSLeay::CTX_new();
    if ( ! $self{ctx} ) {
	$self{ERROR} .= "CTX_new() failed $!\n";
	I2A2::DB::close(\%self);
	return bless \%self, $class;			# ERROR return
    }
    Net::SSLeay::CTX_set_options($self{ctx}, &Net::SSLeay::OP_ALL);
    if ($errs = print_errs("CTX_set_options()")) {	# assignment
	$self{ERROR} .= $errs;
	I2A2::DB::close(\%self);
	return bless \%self, $class;			# ERROR return
    }

    # Turn on VERIFY_PEER
    Net::SSLeay::CTX_set_verify($self{ctx}, &Net::SSLeay::VERIFY_PEER, undef);
    if ($errs = print_errs("CTX_set_verify()")) {	# assignment
	$self{ERROR} .= $errs;
	I2A2::DB::close(\%self);
	return bless \%self, $class;			# ERROR return
    }

    # Set where to look for CA certificates
    if (Net::SSLeay::CTX_load_verify_locations($self{ctx},
		$I2A2::DB::CApath, $I2A2::DB::CAdir) <= 0) {
	$self{ERROR} .= print_errs("CTX_load_verify_locations()");
	I2A2::DB::close(\%self);
	return bless \%self, $class;			# ERROR return
    }
    if (Net::SSLeay::CTX_set_default_verify_paths($self{ctx}) <= 0) {
	$self{ERROR} .= print_errs("CTX_set_default_verify_paths()");
	I2A2::DB::close(\%self);
	return bless \%self, $class;			# ERROR return
    }
    #
    # Set up Client Side certificate if provided
    #
    if ($ClPriv && $ClPub) {
	if (Net::SSLeay::CTX_use_certificate_file($self{ctx}, $ClPub, &Net::SSLeay::FILETYPE_PEM) <= 0) {
	    $self{ERROR} .= print_errs("CTX_use_certificate_file()");
	    I2A2::DB::close(\%self);
	    return bless \%self, $class;		# ERROR return
	}
	if (Net::SSLeay::CTX_use_PrivateKey_file($self{ctx}, $ClPriv, &Net::SSLeay::FILETYPE_PEM) <= 0) {
	    $self{ERROR} .= print_errs("CTX_use_PrivateKey_file()");
	    I2A2::DB::close(\%self);
	    return bless \%self, $class;		# ERROR return
	}
    }

    # Finally, we can open an SSL connection to the server
    $self{ssl} = Net::SSLeay::new($self{ctx});
    if ( ! $self{ssl} ) {
	$self{ERROR} .= "Failed to create SSL $!\n";
	I2A2::DB::close(\%self);
	return bless \%self, $class;			# ERROR return
    }
    Net::SSLeay::set_fd($self{ssl}, fileno($self{socket}));	# Must use fileno
    $resp = Net::SSLeay::connect($self{ssl});
    if ($errs = print_errs("SSL connect()")) {		# assignment
	$self{ERROR} .= $errs;
	I2A2::DB::close(\%self);
	return bless \%self, $class;			# ERROR return
    }

    ### print "Cipher `" . Net::SSLeay::get_cipher($self{ssl}) . "'\n";
    my $cert = Net::SSLeay::get_peer_certificate($self{ssl});
    if ( I2A2::libpuidX::puid_puissuer($cert) ) {
	$self{ERROR} .= "Server Certificate not issued by Purdue Authority.\n";
	I2A2::DB::close(\%self);
	return bless \%self, $class;			# ERROR return
    }

    # Verify the Server's PUID if provided in call
    if ( defined $serverPUID ) {
	$self{puid} = 0;			# avoid uninitialized warning
	if ( I2A2::libpuidX::puid_getcertPUID($cert, $self{puid})
	  || $self{puid} != $serverPUID ) {
	    $self{ERROR} .= "Unable to verify server PUID\n";
	    I2A2::DB::close(\%self);
	    return bless \%self, $class;		# ERROR return
	}
    }

    # Verify the Server's Certificate Serial number if provided in call
    if ( defined $serverCSER ) {
	$self{cser} = 0;
	if ( I2A2::libpuidX::puid_getcertserial($cert, $self{cser})
	  || $self{cser} ne $serverCSER ) {
	    $self{ERROR} .= "Unable to verify cert serial number for server.\n";
	    I2A2::DB::close(\%self);
	    return bless \%self, $class;		# ERROR return
	}
    }

    # Look for the initial "Welcome" message
    $resp = Net::SSLeay::ssl_read_CRLF($self{ssl});	# Perl returns undef on failure
    if ($errs = print_errs("ssl_read_CRLF()")) {	# assignment
	$self{ERROR} .= $errs;
	I2A2::DB::close(\%self);
	return bless \%self, $class;			# ERROR return
    }
    if ( $resp =~ ("^" . &I2A2::PUIDNETD_REPL_WELCOME) ) {
        $self{CONNECTED} = 1;
    } elsif ( $resp =~ ("^" . &I2A2::PUIDNETD_REPL_NAK) ) {
	$self{ERROR} .= "Connection refused\n";
	my %fields = I2A2::DB::DBparse($resp);
	$self{ERROR} .= I2A2::libpuidX::puidnetd_strerror($fields{&I2A2::PUIDNETD_DATA_ERRC})
		if defined $fields{&I2A2::PUIDNETD_DATA_ERRC};
	if (defined $fields{&I2A2::PUIDNETD_DATA_MSG}) {
	    foreach (@{$fields{&I2A2::PUIDNETD_DATA_MSG}}) {
		$self{ERROR} .= $_;
	    }
	}
    } else {
	$self{ERROR} .= "Not a welcome message '$resp'\n";
    }

    return bless \%self, $class;
}


sub request {
# %handle, $req
    my ($self, $req) = @_;

    my ($errs, $line, $resp);

    # See if we're connected

    if ( ! $self->{CONNECTED} ) {
	$self->{ERROR} .= "Not connected.\n";
	return;
    }

    # Issue the request

    $resp = Net::SSLeay::ssl_write_CRLF($self->{ssl}, $req);
    if ($errs = print_errs("ssl_write_CRLF()")) {	# assignment
	$self->{ERROR} .= $errs;
	I2A2::DB::close($self);
	return;						# ERROR return
    }

    $line = Net::SSLeay::ssl_read_CRLF($self->{ssl});	# Perl returns undef on failure
    if ($errs = print_errs("ssl_read_CRLF()")) {	# assignment
	$self->{ERROR} .= $errs;
	I2A2::DB::close($self);
	return;						# ERROR return
    }
    if (1 != chomp $line) { 	# remove \n
	$self->{WARN} .= "no newline found: $line\n";
    }
    if ("\r" ne chop $line) {	# remove \r 
	$self->{WARN} .= "no carriage return found: $line\n";
    }

    # Handle continuation lines

    while (substr($line, -1) eq &I2A2::PUIDNETD_DATA_CONT) {
	my $tmp = Net::SSLeay::ssl_read_CRLF($self->{ssl});	# Perl returns undef on failure
	if ($errs = print_errs("ssl_read_CRLF()")) {	# assignment
	    $self->{ERROR} .= $errs;
	    I2A2::DB::close($self);
	    return;					# ERROR return
	}
	if (substr($tmp, 0, 1) ne &I2A2::PUIDNETD_REPL_CONT) {
	    # should have been a continuation - now we're lost
	    $self->{ERROR} .= "continuation line malformed: $tmp\n";
	    I2A2::DB::close($self);
	    return;
	}
	if (1 != chomp $tmp) { 	# remove \n
	    $self->{WARN} .= "no newline found: $tmp\n";
	}
	if ("\r" ne chop $tmp) {	# remove \r 
	    $self->{WARN} .= "no carriage return found: $tmp\n";
	}
	$line .= substr($tmp, 1);
    }

    return $line;
}


sub close {
    my ($self) = @_;

    my ($errs, $req, $resp, $line);

    # Close the connection

    if ($self->{CONNECTED}) {
	$req = &I2A2::PUIDNETD_CMD_QUIT . &I2A2::PUIDNETD_MSGTERM;
	$resp = Net::SSLeay::ssl_write_CRLF($self->{ssl}, $req);
	if ($errs = print_errs("ssl_write_CRLF()")) {		# assignment
	    $self->{WARN} .= $errs;
	} else {
	    $line = Net::SSLeay::ssl_read_CRLF($self->{ssl});   # Perl returns undef on failure
	    if ($errs = print_errs("ssl_read_CRLF()")) {	# assignment
		$self->{WARN} .= $errs;
	    } else {
		if (1 != chomp $line) { 	# remove \n
		    $self->{WARN} .= "no newline found: $line\n";
		}
		if ("\r" ne chop $line) {	# remove \r 
		    $self->{WARN} .= "no carriage return found: $line\n";
		}
	    }
	}
    }

    $self->{CONNECTED} = 0;		# mark closed
    shutdown $self->{socket}, 1;	# Half close --> No more output, sends EOF to server

    if (defined($self->{ssl})) {
	Net::SSLeay::free($self->{ssl});
	$self->{ssl} = undef;
    }
    if (defined($self->{ctx})) {
        Net::SSLeay::CTX_free($self->{ctx});
	$self->{ctx} = undef;
    }
    close $self->{socket};
}

sub LOG_ERROR {
    my $msg = shift;
    Apache->server->log_error($msg);
}

sub LOG_WARN {
    my $msg = shift;
    Apache->server->warn($msg);
}

sub LOG_STDERR {
    my $msg = shift;
    print STDERR "$msg\n";
}

if (defined $ENV{MOD_PERL}) {
    *ERROR = *LOG_ERROR;
    *WARN = *LOG_WARN;
} else {
    *ERROR = *LOG_STDERR;
    *WARN = *LOG_STDERR;
}


sub log_errors {
    my ($self, $log, $msg) = @_;

    my ($errs);
    if (defined $msg) {
	$msg .= ": ";
    } else {
	$msg = "";
    }
    if ($log && $self->{WARN}) {
	my @m = split "\n", $self->{WARN};
	foreach (@m) {
	    WARN($msg . $_);
	}
    }
    if ($self->{ERROR}) {
	my @m = split "\n", $self->{ERROR};
	foreach (@m) {
	    ERROR($msg . $_) if $log;
	    $errs .= $msg . $_ . "\n";
	}
    }
    $self->{WARN} = undef;
    $self->{ERROR} = undef;
    return $errs;
}


sub DBparse {
    my ($line) = shift;

    my (%hash);

    undef %hash;
    $hash{REPLY} = substr($line, 0, 1);		# save reply character
    %hash = (%hash, DBparseFields(substr($line, 1)));
    return %hash;
}


sub DBparseFields {
    my ($line) = shift;

    my (@fields, %hash);

    @fields = split (&I2A2::PUIDNETD_MSGTERM, $line); 
    if (!defined $fields[0] || $fields[0] eq "") {	# remove empty field
	shift @fields;		# no optional field following reply
    }

    my ($inRec) = 0;
    foreach (@fields) {
        my ($type) = substr($_, 0, 1);
	if ($inRec) {
	    if ($type ne $inRec) {
		my ($tmp) = pop @{$hash{$inRec}};
		$tmp .= $_ . &I2A2::PUIDNETD_MSGTERM;
		push @{$hash{$inRec}}, $tmp;
	    } else {
		$inRec = 0;
	    }
	    next;
	}
        if ($type eq &I2A2::PUIDNETD_DATA_MSG) {
	    push @{$hash{&I2A2::PUIDNETD_DATA_MSG}}, substr($_, 1);
	} elsif (($type eq &I2A2::PUIDNETD_DATA_MDFY) ||
		 ($type eq &I2A2::PUIDNETD_DATA_AUTHC_REC)) {
	    $inRec = $type;
	    my ($tmp) = substr($_, 1);
	    $tmp .= &I2A2::PUIDNETD_MSGTERM
		if (length $tmp);
	    push @{$hash{$inRec}}, $tmp;
	} else {
	    $hash{$type} = substr($_, 1);
	}
    }
    return %hash;
}

1;
__END__

=head1 AUTHOR

Jeff W. Stewart, jws@purdue.edu

=head1 SEE ALSO

=cut
