package I2A2::Authz;

=head1 NAME

I2A2::Authz - I2A2 Authorization routines.

=head1 SYNOPSIS

use I2A2::Authz;

($err, @array) = Authorized($puid, @exp_list);

=head1 DESCRIPTION

The I2A2::Authz module includes procedures which access the I2A2
Authorizer DBM.

=over 4

=cut

require   Exporter;
@ISA    = qw(Exporter);
@EXPORT = qw(Authorized);
@EXPORT_OK = qw();

use strict;

use I2A2;
use I2A2::libpuidX;
use I2A2::DB;


=item	C<($err, @array) = Authorized($puid, @exp_list)>

Authorized() checks each characteristic expression in @exp_list until
it finds an expression that evaluates to true for $puid or runs out
of expressions.

When an expression is found that evaluates to true, Authorized() returns
a value greater than 0, and, when evaulated in list context, the expression.

If no expression in @exp_list evaluates to true, Authorized() returns
the value 0, and the last expression tried (when evaulated in list context).

If an error occurs, Authorized() returns a negative value, and a list
of error messages (when evaulated in list context).

Example:

    use I2A2;
    use I2A2::Authz;
    ...

    @exp_list = ( '0 | 13672', '1 & 1024');
    $puid = 12340;

    if (Authorized($puid, @exp_list) > 0) {
	....	# user is authorized
    }

    ($value, @array) = Authorized($puid, @exp_list);
    if ($value > 0) {
	printf "PUID %s is authorized by '%s'\n",
	    Format_PUID($puid), $array[0];
    } elsif ($value == 0) {
	printf "PUID %s is NOT authorized\n", Format_PUID($puid);
    } else {
	printf "An error occured:\n", $puid;
	foreach $msg (@array) {
	    printf "\t%s\n", $msg;
	}
    }

=cut

sub Authorized {
    my ($puid, @list) = @_;
    my ($err, $errmsg, @msg, $line, $req, %fields);
    my ($spec, $value);

    # This works the hard way by opening a connection for every lookup.
    # Open a connection to the server.

    my $AuthZ = I2A2::DB->connect(&I2A2::PUIDNETD_HOST_AUTHZ,
		    &I2A2::PUIDNETD_SVC_SSL_AUTHZ, &I2A2::PUIDNETD_AUTHZ_PUID,
		    &I2A2::PUIDNETD_AUTHZ_CSER);
    if ($errmsg = $AuthZ->log_errors(1, "AuthZ connect")) {	# assignment
	$err = -1;
	push @msg, "Authorizer connect failed.";
	goto EXIT;
    }

    #
    # Loop through the specification list until one succeeds.
    #

    for (@list) {

	# Build the request

	$err = -1;
	undef $value;
	$spec = $_;
	$req = &I2A2::PUIDNETD_CMD_LOOKUP;
	$req .= &I2A2::PUIDNETD_DATA_PUID . $puid . &I2A2::PUIDNETD_MSGTERM;
	$req .= &I2A2::PUIDNETD_DATA_AUTHZ_EXP . $spec . &I2A2::PUIDNETD_MSGTERM;

	# Issue the request

	$line = $AuthZ->request($req);
	if ($errmsg = $AuthZ->log_errors(1, "AuthZ request")) {	# assignment
	    push @msg, split "\n", $errmsg;
	    goto EXIT;
	}

	%fields = I2A2::DB::DBparse($line); 

	if ($fields{REPLY} eq &I2A2::PUIDNETD_REPL_ACK) {
	    $err = 0;
	    $value = $fields{&I2A2::PUIDNETD_DATA_AUTHZ_EXPV}+0; # force int
	    last if ($value);
	} elsif ($fields{REPLY} eq &I2A2::PUIDNETD_REPL_NAK) {
	    $err = -1;
	    @msg = @{$fields{&I2A2::PUIDNETD_DATA_MSG}};
	    unshift @msg, "Error number " . $fields{&I2A2::PUIDNETD_DATA_ERRC}
		if (defined $fields{&I2A2::PUIDNETD_DATA_ERRC});
	} else {
	    $err = -2;
	    push @msg, "unexpected answer: '$line'.";
	}
    }

    # Close the connection

    EXIT:
    if (defined $AuthZ) {
	$AuthZ->close();
	undef $AuthZ;
    }

    if ($err) {
	if (wantarray) {
	    return ($err, @msg);
	} else {
	    return $err;
	}
    } else {
	if (wantarray) {
	    return ($value, $spec);
	} else {
	    return $value;
	}
    }
}

1;
__END__

=head1 NOTES

Authorized() evaluates the expressions in @exp_list looking for one that
evaluates to true.  Any errors within the expressions, such as syntax
errors or characteristic numbers that do not exist, are not reported
except for the last expression checked.

=head1 AUTHOR

Jeff W. Stewart, jws@purdue.edu

=head1 SEE ALSO

I2A2(3)

=cut
