package I2A2::Authc;

=head1 NAME

I2A2::Authc - I2A2 Authentication routines.

=head1 SYNOPSIS

use I2A2::Authc;

($err, @array) = Authenticate($id, $passwd, $realm);

$line = RealmLookup($id, $realm);

=head1 DESCRIPTION

The I2A2::Authc module includes procedures which access the I2A2 Authenticator
DBM.

=over 4

=cut

require   Exporter;
@ISA    = qw(Exporter);
@EXPORT = qw(Authenticate RealmLookup);
@EXPORT_OK = qw();

use strict;

use I2A2;
use I2A2::libpuidX;
use I2A2::DB;
use MIME::Base64 qw(encode_base64);


=item C<($err, @array) = Authenticate($id, $passwd, $realm)>

Authenticate() trys to authenticate the user ($id can be a PUID or an alias)
with the given password in the given realm.

When authentication succeeds, $err will be zero (0) and @array will
contain the PUID, alias, and name of the user.

On failure, $err will be non-zero and @array will contain strings
explaining the error.

Example:

    use I2A2;
    use I2A2::Authc;
    ...

    ($err, @array) = Authenticate("usera", "password", "Purdue");
    if ($err) {
	foreach (@array) {
	    print STDERR "$_\n";
	}
	...	# return error or die, etc.
    }
    ($puid, $alias, $name) = @array;
    ...

=cut

sub Authenticate {
    my ($id, $passwd, $realm) = @_;
    my ($b64passwd, $err, $errmsg, $line, $req, $rFld, $rMsg);

    $err = -1;
    $b64passwd = encode_base64($passwd, '');
    $req = &I2A2::PUIDNETD_CMD_AUTHC;
    if ($id =~ /^\d[-\d]+$/) {		# all digits and "-"  (PUID)
	$id =~ s/-//g;			# remove "-" characters
	$req .= &I2A2::PUIDNETD_DATA_PUID . $id . &I2A2::PUIDNETD_MSGTERM;
	if (luhnck($id)) {
	    $err = -3;
	    push @$rMsg, "Invalid PUID.";
	    goto EXIT;
	}
    } elsif ($id =~ /[\s*]/) {		# contains a space (Name)
	$req .= &I2A2::PUIDNETD_DATA_CNM . $id . &I2A2::PUIDNETD_MSGTERM;
    } else {				# (alias)
	$req .= &I2A2::PUIDNETD_DATA_AKA . $id . &I2A2::PUIDNETD_MSGTERM;
    }
    $req .= &I2A2::PUIDNETD_DATA_AUTHC_REC . &I2A2::PUIDNETD_MSGTERM;
    $req .= &I2A2::PUIDNETD_DATA_AUTHC_RNAME . $realm . &I2A2::PUIDNETD_MSGTERM;
    $req .= &I2A2::PUIDNETD_DATA_AUTHC_PWD . $b64passwd . &I2A2::PUIDNETD_MSGTERM;
    $req .= &I2A2::PUIDNETD_DATA_AUTHC_REC . &I2A2::PUIDNETD_MSGTERM;

    # This works the hard way by opening a connection for every lookup.
    # Open a connection to the server.

    my $AuthC = I2A2::DB->connect(&I2A2::PUIDNETD_HOST_AUTHC,
		    &I2A2::PUIDNETD_SVC_SSL_AUTHC, &I2A2::PUIDNETD_AUTHC_PUID,
		    &I2A2::PUIDNETD_AUTHC_CSER);
    if ($errmsg = $AuthC->log_errors(1, "AuthC connect")) {	# assignment
	push @$rMsg, $errmsg;
	goto EXIT;
    }

    # Issue the request

    $line = $AuthC->request($req);
    if ($errmsg = $AuthC->log_errors(1, "AuthC authenticate")) { # assignment
	push @$rMsg, $errmsg;
	goto EXIT;
    }

    ($err, $rMsg, $rFld) = check_reply($line);

    #
    # Special case for testing - NOT FOR PRODUCTION USE - JWS
    #
####if ($err != 0) {
####	if ($passwd eq "pick a password to allow testing") {
####	my $sep = &I2A2::PUIDNETD_DATA_AUTHC_REC;
####	    # convert authenticate request to a lookup request
####	    substr($req, 0, 1) = &I2A2::PUIDNETD_CMD_LOOKUP;
####	    $req =~ s/$sep.*//;		# truncate request at Realm Record
####	    # Issue the request
####	    $line = $AuthC->request($req);
####	    if ($errmsg = $AuthC->log_errors(1, "AuthC lookup")) { # assignment
####		push @$rMsg, $errmsg;
####		goto EXIT;
####	    }
####	    ($err, $rMsg, $rFld) = check_reply($line);
####	}
####}
    # End of Special case - NOT FOR PRODUCTION USE - JWS

    # Close the connection

    EXIT:
    if (defined $AuthC) {
	$AuthC->close();
	undef $AuthC;
    }

    if (wantarray) {
	if ($err) {
	    return ($err, @$rMsg);
	} else {
	    return ($err,
		$rFld->{&I2A2::PUIDNETD_DATA_PUID}+0,		# puid
		$rFld->{&I2A2::PUIDNETD_DATA_AKA},		# alias
		$rFld->{&I2A2::PUIDNETD_DATA_CNM});		# name
	}
    } else {
	return $err;
    }
}


=item C<$line = RealmLookup($id, $realm)>

RealmLookup() requests an Authenticator lookup for $id (PUID or alias)
in $realm and returns the response from the Authenticator DBM (or an
error response in the same format).

Example:

    use I2A2;
    use I2A2::Authc;
    use I2A2::DB;
    ...

    $line = RealmLookup("usera", "Purdue");
    %fields = DBparse($line);
    if ($fields{REPLY} eq &I2A2::PUIDNETD_REPL_ACK) {
	...	# Replied with ACK
    } else {
	...	# error code here
    }
    ...

=cut

sub RealmLookup {
    my ($id, $realm) = @_;
    my ($errmsg, $req, $line);

    $req = &I2A2::PUIDNETD_CMD_LOOKUP;
    if ($id =~ /^\d[-\d]+$/) {          # all digits and "-"  (PUID)
        $id =~ s/-//g;                  # remove "-" characters
        $req .= &I2A2::PUIDNETD_DATA_PUID . $id . &I2A2::PUIDNETD_MSGTERM;
        if (luhnck($id)) {
	    # build up a NAK message
	    $line = &I2A2::PUIDNETD_REPL_NAK . &I2A2::PUIDNETD_MSGTERM;
	    $line .= &I2A2::PUIDNETD_DATA_MSG . "Invalid PUID";
	    $line .= &I2A2::PUIDNETD_MSGTERM;
            return $line;
        }
    } elsif ($id =~ /[\s*]/) {          # contains a space (Name)
        $req .= &I2A2::PUIDNETD_DATA_CNM . $id . &I2A2::PUIDNETD_MSGTERM;
    } else {                            # (alias)
        $req .= &I2A2::PUIDNETD_DATA_AKA . $id . &I2A2::PUIDNETD_MSGTERM;
    }

    $req .= &I2A2::PUIDNETD_DATA_AUTHC_REC;
    $req .= &I2A2::PUIDNETD_DATA_AUTHC_RNAME . $realm . &I2A2::PUIDNETD_MSGTERM;
    $req .= &I2A2::PUIDNETD_DATA_AUTHC_REC;

    # This works the hard way by opening a connection for every lookup.
    # Open a connection to the server.

    my $AuthC = I2A2::DB->connect(&I2A2::PUIDNETD_HOST_AUTHC,
		    &I2A2::PUIDNETD_SVC_SSL_AUTHC, &I2A2::PUIDNETD_AUTHC_PUID,
		    &I2A2::PUIDNETD_AUTHC_CSER);
    if ($errmsg = $AuthC->log_errors(1, "RealmLookup connect") ) {	# assignment
	# build up a NAK message with the errors
	my $trm_msg = &I2A2::PUIDNETD_MSGTERM . &I2A2::PUIDNETD_DATA_MSG;
	chomp $errmsg;				# remove trailing newline
	$errmsg =~ s/\n/$trm_msg/go;		# change newline to msgterm/msg
	$line = &I2A2::PUIDNETD_REPL_NAK . $trm_msg;
	$line .= $errmsg;
	$line .= &I2A2::PUIDNETD_MSGTERM;
    } else {
	# Issue the request

	$line = $AuthC->request($req);
        if ($errmsg = $AuthC->log_errors(1, "RealmLookup request")) {	# assignment
	    # build up a NAK message with the errors
	    my $trm_msg = &I2A2::PUIDNETD_MSGTERM . &I2A2::PUIDNETD_DATA_MSG;
	    chomp $errmsg;			# remove trailing newline
	    $errmsg =~ s/\n/$trm_msg/go;	# change newline to msgterm/msg
	    $line = &I2A2::PUIDNETD_REPL_NAK . $trm_msg;
	    $line .= $errmsg;
	    $line .= &I2A2::PUIDNETD_MSGTERM;
	}

	# Close the connection

	$AuthC->close();
	undef $AuthC;
    }

    return $line;
}


##
# ($err, $rMsg, $rFld) = check_reply($line);
#	$line	- line returned by AuthC request method
#    output:
#	$err	- an error number (0 if OK)
#	$rMsg	- Array reference to error messages
#	$rFld	- Hash reference to fields (from DBparse())
#

sub check_reply {
    my ($line) = @_;
    my (@msg, %fields);
    my $err = -1;

    %fields = I2A2::DB::DBparse($line);
    if ($fields{REPLY} eq &I2A2::PUIDNETD_REPL_ACK) {
	$err = 0;
    } elsif ($fields{REPLY} eq &I2A2::PUIDNETD_REPL_NAK) {
	$err = $fields{&I2A2::PUIDNETD_DATA_ERRC}
	    if ($fields{&I2A2::PUIDNETD_DATA_ERRC});
	if ($fields{&I2A2::PUIDNETD_DATA_MSG}) {
	    @msg = @{$fields{&I2A2::PUIDNETD_DATA_MSG}};
	} else {
	    push @msg, "Authentication failed.";
	}
    } else {
	$err = -2;
	push @msg, "unexpected answer: '$line'.";
    }
    if (wantarray) {
	return ($err, \@msg, \%fields);
    } else {
	return $err;
    }
}


1;
__END__

=back

=head1 AUTHOR

Jeff W. Stewart, jws@purdue.edu

=head1 SEE ALSO

I2A2(3), I2A2::DB(3)

=cut
