package I2A2::Access;

=head1 NAME

I2A2::Access - ...

=head1 SYNOPSIS

 PerlSetVar  TicketSecret    <secret_text_path>

 <Directory ...>
     SetHandler    perl-script
     PerlAccessHandler   I2A2::Access

     AllowUser     <user list>
     DenyUser      <user list>
     CharsExp      <characteristic expression>
 </Directory>

=head1 DESCRIPTION

...

=cut

require   Exporter;
@ISA    = qw(Exporter);
@EXPORT = qw(DIR_MERGE SERVER_CREATE);
@EXPORT_OK = qw();

use strict;
use vars qw($VERSION);

use Apache::Constants qw(:response :http);
use Apache::ModuleConfig ();
use Apache::Log;
use CGI::Cookie ();
use DynaLoader ();

## This file is processed by Apache::ExtUtils::command_table() in
#  Makefile.PL to create the Access.xs file.  However, the I2A2
#  modules aren't available when that happens.  So, process those
#  modules in an "eval" to trap the error.  Don't fail during the
#  configure process.
eval "
    use I2A2;
    use I2A2::TicketTool ();
    use I2A2::Authz;
    use I2A2::DB;
";
die "$@" if $@ && ! defined($ENV{PERL_MAKEFILE_PL});

$VERSION = '1.00';

if($ENV{MOD_PERL}) {
    no strict;
    @ISA = qw(DynaLoader);
    __PACKAGE__->bootstrap($VERSION);
}


sub new {
    return bless {}, shift;
}


sub handler {
    my $r = shift;
    my ($result, $msg, $expl, $info);

    my $ticketTool = I2A2::TicketTool->new($r);
    ($result, $msg, $info) = $ticketTool->verify_ticket($r);
    if ($result == OK) {
	($result, $msg, $expl) = ok_access($r, $info);
    }
    if ($result == AUTH_REQUIRED) {
	$r->log_reason($msg, $r->filename);
	my $cookie = $ticketTool->make_return_address($r);
	my $url = $ticketTool->loginURL;
	unless ($url) {
	    $r->log_error("Couldn't get URL path for login");
	    return SERVER_ERROR;
	}
	$r->err_headers_out->add('Set-Cookie' => $cookie);

	# build Realm List for use by ticketLogin
        my $cfg = Apache::ModuleConfig->get($r);
	my %list;
	if (defined $cfg->{Realms}{ANY}) {
	    %list = %I2A2::Access::AllRealms;
	} else {
	    %list = %{$cfg->{Realms}};
	}
	foreach (keys %{$cfg->{RealmAdd}}) {
	    $list{$_} = $cfg->{RealmAdd}{$_};
	}
	foreach (keys %{$cfg->{RealmDel}}) {
	    delete $list{$_};
	}
	my $list = join(",", sort values %list);
	my $def = $cfg->{DefaultRealm} || "Purdue";
	$r->err_headers_out->add('Location' => "$url?realmlist=$def,$list");
	return REDIRECT;
    }
    unless ($result == OK) {
	$r->custom_response(FORBIDDEN, $expl);
	$r->log_reason($msg, $r->filename);
	return FORBIDDEN;
    }

    return OK;
}


# ($result, $msg, $expl) = ok_access($r, $ticketInfo);
#    returns $result of OK, AUTH_REQUIRED, FORBIDDEN
sub ok_access {
    my ($r, $ticketInfo) = @_;
    my ($ok);
    return (AUTH_REQUIRED, 'malformed ticket', '')
	unless $ticketInfo->{'user'}
	    && $ticketInfo->{'puid'}
	    && $ticketInfo->{'realm'};
    my $alias = $ticketInfo->{'user'};
    my $puid = $ticketInfo->{'puid'};
    my $realm = lc $ticketInfo->{'realm'};

    my $cfg = Apache::ModuleConfig->get($r);
    my $RejectRealm = 1;
    if (! defined $cfg->{RealmDel}{$realm}) {
	if (defined $cfg->{Realms}{ANY}) {
	    $RejectRealm = 0;
	} elsif (defined $cfg->{Realms}{$realm}) {
	    $RejectRealm = 0;
	} elsif (defined $cfg->{RealmAdd}{$realm}) {
	    $RejectRealm = 0;
	}
    }
    if ($RejectRealm) {
	return(AUTH_REQUIRED,
	    'Authentication in a different realm is required', '?');
    }

    my $explanation = <<END;
<TITLE>Unauthorized</Title>
<H1>You are not authorized to access this page</H1>
<UL>
END
    my $allow = 0;
    my $deny = 0;

    foreach ($alias, $puid) {
	if (defined $cfg->{Access}{$_}) {
	    my $val = $cfg->{Access}{$_};
	    $allow |= ($val > 0);
	    $deny |= ($val < 0);
	}
    }
    if ($deny) {				# Access denied?
	$explanation .= "<LI>Denied by configuration.\n";
	$explanation .= "</UL>\n";
	return (FORBIDDEN, "user $alias: not authorized", $explanation);
    }
    if ($allow) {				# Access allowed?
	return (OK, 'ok', '');
    }

    # Try the Characteristic Expressions

    if (defined (@{$cfg->{Access}{CHARS_EXP}})) {
	my @charList = @{$cfg->{Access}{CHARS_EXP}};
	return (OK, 'ok', '') if (Authorized($puid, @charList) > 0);
	$explanation .= "<LI>Failed to match Characteristics.\n";
    } else {
	$explanation .= "<LI>No permission for '$alias' granted in configuration.\n";
    }

    $explanation .= "</UL>";

    return (FORBIDDEN, "user $alias: not authorized", $explanation);
}


###
# Handle Apache Configuration directives: AllowUser, DenyUser, CharsExp.
#

sub DefaultRealm ($$$) {
    my ($cfg, $params, $realm) = @_;

    $cfg->{DefaultRealm} = $realm;
}


sub Realms ($$@) {
    my ($cfg, $params, $realm) = @_;
    my ($type, $string);

    if (lc($realm) eq 'any') {
	if (defined $cfg->{Realms}) {
###	    ERROR
	}
	$cfg->{Realms}{ANY} = 1;
	delete $cfg->{RealmsAdd};	# shouldn't exist - error if they do?
	delete $cfg->{RealmsDel};	# shouldn't exist - error if they do?
    } else {
	$realm =~ m/([+-]?)(.*)/;
	$type = $1;
	$string = $2;
	$realm = lc($string);

	# This could be an error, but add new realms to the list of all realms
	# if we're not reading a .htaccess file.
	# It's possible that "%AllRealms" didn't get built correctly.  JWS
	if (! defined $I2A2::Access::AllRealms{$realm}
	  && ($Apache::Server::Starting || $Apache::Server::ReStarting)) {
	    $I2A2::Access::AllRealms{$realm} = $string;
	}
	# Check that a realm isn't mentioned multiple times?
###	ERROR if defined $cfg->{Realms}{$realm}
###	    || defined $cfg->{RealmAdd}{$r}
###	    || defined $cfg->{RealmDel}{$r}
###	    || defined $cfg->{Realms}{ANY} && $type ne '-';
	if ($type eq '+') {
	    $cfg->{RealmAdd}{$realm} = $string;
	} elsif ($type eq '-') {
	    $cfg->{RealmDel}{$realm} = $string;
	} else {
	    $cfg->{Realms}{$realm} = $string;
	}
    }
}


sub AllowUser ($$@) {
    my ($cfg, $params, $id) = @_;
    if ($id =~ /^clear[_-]list$/i) {	# clear list
	foreach (keys %{$cfg->{Access}}) {
	    if ($cfg->{Access}{$_} > 0) {
		delete $cfg->{Access}{$_};	# remove entry
	    }
	}
	$cfg->{CLEAR_ALLOW_LIST} = 1;
	return;
    } elsif ($id =~ /^\d[-\d]+$/) {	# all digits and "-"  (PUID)
	$id =~ s/-//g;			# remove "-" characters
	$id += 0;			# remove leading zeros
    }
    $cfg->{Access}{$id} = 1;
}


sub DenyUser ($$@) {
    my ($cfg, $params, $id) = @_;
    if ($id =~ /^clear[_-]list$/i) {	# clear list
	foreach (keys %{$cfg->{Access}}) {
	    if ($cfg->{Access}{$_} < 0) {
		delete $cfg->{Access}{$_};	# remove entry
	    }
	}
	$cfg->{CLEAR_DENY_LIST} = 1;
	return;
    } elsif ($id =~ /^\d[-\d]+$/) {	# all digits and "-"  (PUID)
	$id =~ s/-//g;			# remove "-" characters
	$id += 0;			# remove leading zeros
    }
    $cfg->{Access}{$id} = -1;
}


sub CharsExp ($$$;*) {
    my ($cfg, $params, $exp, $cfg_fh) = @_;
    $exp =~ s/\t/ /g;
    push @{$cfg->{Access}{CHARS_EXP}}, $exp;
    return 0;					#JWS test return value
}


sub SERVER_CREATE {
    my $class = shift;

    my ($line);

    if (!defined $I2A2::Access::BldTime || time > $I2A2::Access::BldTime + 60) {
	my $AuthC = I2A2::DB->connect(&I2A2::PUIDNETD_HOST_AUTHC,
	    &I2A2::PUIDNETD_SVC_SSL_AUTHC, &I2A2::PUIDNETD_AUTHC_PUID,
	    &I2A2::PUIDNETD_AUTHC_CSER );
	if ($AuthC->log_errors(1, "SERVER_CREATE connect")) {	# assignment	
	    # build a NAK to cause defaults to be used
	    $line = &I2A2::PUIDNETD_REPL_NAK . &I2A2::PUIDNETD_MSGTERM;
	} else {
	    my $req  = &I2A2::PUIDNETD_CMD_GETINFO . &I2A2::PUIDNETD_DATA_MSG
		. &I2A2::PUIDNETD_GIFO_REALMS . &I2A2::PUIDNETD_MSGTERM;

	    $line = $AuthC->request($req);
	    if ($AuthC->log_errors(1, "SERVER_CREATE request")) {	# assignment	
		# build a NAK to cause defaults to be used
		$line = &I2A2::PUIDNETD_REPL_NAK . &I2A2::PUIDNETD_MSGTERM;
	    }
	    $AuthC->close;
	    undef $AuthC;
	}

	my %ans = I2A2::DB::DBparse($line);
	my @list = ();
	if ( $ans{REPLY} eq &I2A2::PUIDNETD_REPL_ACK ) {
	    foreach (@{$ans{&I2A2::PUIDNETD_DATA_MSG}}) {
		if ( /^Name: ([^;]+);/ ) {
		    push @list, $1;
		}
	    }
	    foreach (@list) {
		($I2A2::Access::AllRealms{$_} = $_) =~ s/(\w+)/ ucfirst($1) /eg;
	    }
	    $I2A2::Access::BldTime = time;
	    my $msg = "Build realm list => " . join (", ", values %I2A2::Access::AllRealms);
	    Apache->server->log->info($msg);
	} else {
	    if (!defined %I2A2::Access::AllRealms) {
		%I2A2::Access::AllRealms = ('purdue' => 'Purdue');
	    }
	    Apache->server->warn("Couldn't build realm list");
	}
    }

    bless {}, $class;
}


sub DIR_MERGE {
    my ($parent, $current) = @_;
    my %new = (%$current);		# Start with current

    # override the "Access" restrictions
    $new{Access} = {};			# clear the Access list
#    %{$new{Access}} = (%{$parent->{Access}}, %{$current->{Access}});
#    %{$new{Access}} = %{$parent->{Access}};
    my $clear_allow = defined $current->{CLEAR_ALLOW_LIST};
    my $clear_deny = defined $current->{CLEAR_DENY_LIST};
    foreach (keys %{$parent->{Access}}) {
	next if ($parent->{Access}{$_} > 0 && $clear_allow);
	next if ($parent->{Access}{$_} < 0 && $clear_deny);
	$new{Access}{$_} = $parent->{Access}{$_};
    }
    %{$new{Access}} = (%{$new{Access}}, %{$current->{Access}})
	if defined $current->{Access};
    delete $new{CLEAR_ALLOW_LIST};	# remove flag
    delete $new{CLEAR_DENY_LIST};	# remove flag

    # Realms
    if (!defined $new{DefaultRealm} && defined $parent->{DefaultRealm}) {
	$new{DefaultRealm} = $parent->{DefaultRealm};
    }
    delete $new{Realms};
    delete $new{RealmAdd};
    delete $new{RealmDel};
    if (defined $current->{Realms}) {
	$new{Realms} = $current->{Realms};
	$new{RealmAdd} = $current->{RealmAdd} if defined $current->{RealmAdd};
	$new{RealmDel} = $current->{RealmDel} if defined $current->{RealmDel};
    } else {
	if (defined $parent->{Realms}) {
	    $new{Realms} = $parent->{Realms};
	} else {
##	    $new{Realms} = {};
	    $new{Realms}{ANY} = 1;
	}
	my $any = defined $new{Realms}{ANY};
##	if (!$any &&
##	  (defined $current->{RealmAdd} || defined $parent->{RealmAdd})) {
##	    $new{RealmAdd} = {};
##	}
##	if (defined $current->{RealmDel} || defined $parent->{RealmDel}) {
##	    $new{RealmDel} = {};
##	}
	if (!$any && defined $parent->{RealmAdd}) {
	    for (keys %{$parent->{RealmAdd}}) {
		$new{RealmAdd}{$_} = $parent->{RealmAdd}{$_};
	    }
	}
	if (defined $parent->{RealmDel}) {
	    for (keys %{$parent->{RealmDel}}) {
		$new{RealmDel}{$_} = $parent->{RealmDel}{$_};
	    }
	}
	# Add any new realms (and remove from delete list)
	if (defined $current->{RealmAdd}) {
	    for (keys %{$current->{RealmAdd}}) {
		delete $new{RealmDel}{$_}
		    if defined $new{RealmDel}{$_};
		$new{RealmAdd}{$_} = $current->{RealmAdd}{$_}
		    if defined !$any;
	    }
	}
	# now add realms to remove (and remove from add list)
	if (defined $current->{RealmDel}) {
	    for (keys %{$current->{RealmDel}}) {
		delete $new{RealmAdd}{$_}
		    if defined $new{RealmAdd}{$_};
		$new{RealmDel}{$_} = $current->{RealmDel}{$_};
	    }
	}
	if (keys %{$new{RealmAdd}} == 0) {
	    delete $new{RealmAdd};
	}
	if (keys %{$new{RealmDel}} == 0) {
	    delete $new{RealmDel};
	}
    }

##    open TTY, ">/dev/tty" or die "Can't open tty\n";
##    print TTY "Incoming Parent:\n" . dump_cfg_tbl($parent);
##    print TTY "Incoming Current:\n" . dump_cfg_tbl($current);
##    print TTY "Merged:\n" . dump_cfg_tbl(\%new);
##    print TTY "\n\n";
##    close TTY;

    return bless \%new, ref($parent);
}


sub dump_cfg_tbl {
    my $ref = shift;
    my ($str, @exp);

    my $ra = $ref->{Access};

    if ($ref->{DefaultRealm}) {
	$str .= "    DefaultRealm -> " . $ref->{DefaultRealm} . "\n";
    }
    if ($ref->{Realms}) {
	$str .= "    Realms -> " . join(" ", keys %{$ref->{Realms}}) . "\n";
    }
    if ($ref->{RealmAdd}) {
	$str .= "    RealmAdd -> " . join(" ", keys %{$ref->{RealmAdd}}) . "\n";
    }
    if ($ref->{RealmDel}) {
	$str .= "    RealmDel -> " . join(" ", keys %{$ref->{RealmDel}}) . "\n";
    }
    if ($ref->{CLEAR_ALLOW_LIST}) {
	$str .= "    CLEAR_ALLOW_LIST\n";
    }
    if ($ref->{CLEAR_DENY_LIST}) {
	$str .= "    CLEAR_DENY_LIST\n";
    }
    if (defined %$ra) {
	foreach (sort keys %$ra) {
	    if ($_ eq "CHARS_EXP") {
		@exp = @{$ra->{$_}};
	    } else {
		$str .= "    $_ -> " . $ra->{$_} . "\n";
	    }
	}
    }
    if (scalar(@exp)) {
	$str .= "    CHARS_EXP ->\n";
	foreach (@{$ra->{CHARS_EXP}} ) {
	    $str .= "\t$_\n";
	}
    }
    return $str;
}


sub dump_cfg {
    my $ref = shift;
    my ($str);
    my (@allow, @deny, @exp, %realms);

    my $ra = $ref->{Access};

    $str = "AllRealms -> " . join(" ", sort values %I2A2::Access::AllRealms)
	. "\n\n";


    if ($ref->{DefaultRealm}) {
	$str .= "DefaultRealm\t" . $ref->{DefaultRealm} . "\n";
    }
##    $str .= "Realms\t";
##    if (defined $ref->{Realms}) {
##	if (defined $ref->{Realms}{ANY}) {
##	    $str .= "ANY";
##	} else {
##	    $str .= join(" ", sort values %{$ref->{Realms}});
##	}
##    }
##    if ($ref->{RealmAdd}) {
##	$str .= join(" +", "", sort values %{$ref->{RealmAdd}});
##    }
##    if ($ref->{RealmDel}) {
##	$str .= join(" -", "", sort values %{$ref->{RealmDel}});
##    }
##    $str .= "\n";

    if ( $ref->{Realms} ) {
	%realms = ( %{$ref->{Realms}} );
    } else {
	%realms = ();
    }
    if (defined $realms{ANY}) {
	delete $realms{ANY};
    } else {
	if ($ref->{RealmAdd}) {
	    foreach (keys %{$ref->{RealmAdd}}) {
		$realms{$_} = $ref->{RealmAdd}{$_};
	    }
	}
    }
    if ($ref->{RealmDel}) {
	foreach (keys %{$ref->{RealmDel}}) {
	    delete $realms{$_};
	}
    }
    $str .= "Realms\t";
    if (defined $ref->{Realms}{ANY}) {
	$str .= "ANY";
	if ($ref->{RealmDel}) {
	    $str .= join(" -", "", values %{$ref->{RealmDel}});
	}
    } else {
        $str .= join(" ", values %realms);
    }
    $str .= "\n";


    foreach (sort keys %$ra) {
	if ($_ eq "CHARS_EXP") {
	    @exp = @{$ra->{$_}};
	} elsif ($ra->{$_} > 0) {
	    push @allow, $_;
	} else {
	    push @deny, $_;
	}
    }
    $str .= "AllowUser\t" . join(" ", @allow) . "\n" if scalar(@allow);
    $str .= "DenyUser\t" . join(" ", @deny) . "\n" if scalar(@deny);
    if (defined @exp) {
	foreach (@exp) {
	    $str .= "CharsExp\t$_\n";
	}
    }
    return $str;
}


sub dump {
    my $r = shift;
    my $type = "";

    $r->content_type("text/html");
    $r->send_http_header;

    my $loc = $r->location;			# get <Location dir>
    (my $uri = $r->uri) =~ s/$loc$//;		# build new uri
    my $subr = $r->lookup_uri($uri);		# process subrequest
    my $file = $subr->filename;
    my $status = $subr->status;
    my $content_type = $subr->content_type;
    if ($status == HTTP_OK && ! -e $file) {
	$status = HTTP_NOT_FOUND;
    }
    if (-d $file) {
	$type = "(Directory)";
    }

    my $cfg = Apache::ModuleConfig->get($subr);	# get the configuration
    my $str = dump_cfg($cfg);			# printable form

    my $text = <<"EOF";
<HTML>
<HEAD>
</HEAD>
<BODY>
<H3>Status = $status</H3>
Content Type: $content_type
<H4>Access restrictions for <i>$uri</i>:</H4>
<P> &nbsp; &nbsp; $file <B>$type</B>
<pre>
$str
</pre>
</BODY>
</HTML>
EOF

    $r->print($text);
    return OK;
}

1;
__END__

=head1 AUTHOR

Jeff W. Stewart, jws@purdue.edu

=head1 SEE ALSO

=cut
